import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { TEMPLATES } from '@/data/templates';
import { Laptop, Heart, Settings, Briefcase, ArrowRight, Check } from 'lucide-react';

interface TemplatesProps {
  onSelectTemplate: (templateId: string) => void;
}

const industryIcons: Record<string, React.ReactNode> = {
  tech: <Laptop className="h-5 w-5" />,
  healthcare: <Heart className="h-5 w-5" />,
  engineering: <Settings className="h-5 w-5" />,
  business: <Briefcase className="h-5 w-5" />,
};

const visaBadges: Record<string, string> = {
  tech: 'Skilled Worker Visa',
  healthcare: 'Health & Care Visa',
  engineering: 'Skilled Worker Visa',
  business: 'Skilled Worker Visa',
};

export function Templates({ onSelectTemplate }: TemplatesProps) {
  const featuredTemplates = TEMPLATES.slice(0, 6);

  return (
    <section className="py-16 md:py-24">
      <div className="container mx-auto px-4">
        <div className="text-center max-w-3xl mx-auto mb-12">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Industry-Specific CV Templates
          </h2>
          <p className="text-lg text-muted-foreground">
            Choose from our collection of recruiter-approved templates designed for UK visa sponsorship applications. 
            Each template includes role-specific keywords and formatting.
          </p>
        </div>

        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
          {featuredTemplates.map((template) => (
            <Card key={template.id} className="group hover:shadow-xl transition-all duration-300">
              <CardHeader>
                <div className="flex items-center justify-between mb-2">
                  <div className="p-2 bg-primary/10 rounded-lg group-hover:bg-primary/20 transition-colors">
                    {industryIcons[template.industry]}
                  </div>
                  <Badge variant="outline" className="text-xs">
                    {visaBadges[template.industry]}
                  </Badge>
                </div>
                <CardTitle className="text-xl">{template.name}</CardTitle>
                <CardDescription>{template.description}</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <p className="text-sm font-medium">Includes:</p>
                  <ul className="space-y-1">
                    {template.keywords.slice(0, 3).map((keyword, idx) => (
                      <li key={idx} className="flex items-center gap-2 text-sm text-muted-foreground">
                        <Check className="h-3.5 w-3.5 text-green-500" />
                        {keyword}
                      </li>
                    ))}
                  </ul>
                </div>
                <Button
                  onClick={() => onSelectTemplate(template.id)}
                  className="w-full gap-2"
                  variant="outline"
                >
                  Use Template
                  <ArrowRight className="h-4 w-4" />
                </Button>
              </CardContent>
            </Card>
          ))}
        </div>

        <div className="text-center mt-10">
          <Button size="lg" variant="outline" className="gap-2">
            View All Templates
            <ArrowRight className="h-4 w-4" />
          </Button>
        </div>
      </div>
    </section>
  );
}
