import { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { TEMPLATES, getExampleCV } from '@/data/templates';
import type { CVData } from '@/types/cv';
import { INDUSTRIES, EXPERIENCE_LEVELS } from '@/types/cv';
import { Laptop, Heart, Settings, Briefcase, GraduationCap, Sparkles } from 'lucide-react';

interface TemplateSelectorProps {
  onSelectTemplate: (template: typeof TEMPLATES[0], exampleData: CVData) => void;
}

const industryIcons: Record<string, React.ReactNode> = {
  tech: <Laptop className="h-5 w-5" />,
  healthcare: <Heart className="h-5 w-5" />,
  engineering: <Settings className="h-5 w-5" />,
  business: <Briefcase className="h-5 w-5" />,
  education: <GraduationCap className="h-5 w-5" />,
};

export function TemplateSelector({ onSelectTemplate }: TemplateSelectorProps) {
  const [selectedIndustry, setSelectedIndustry] = useState<string | null>(null);
  const [selectedLevel, setSelectedLevel] = useState<string | null>(null);

  const filteredTemplates = TEMPLATES.filter((template) => {
    if (selectedIndustry && template.industry !== selectedIndustry) return false;
    if (selectedLevel && template.experienceLevel !== selectedLevel) return false;
    return true;
  });

  const handleSelectTemplate = (template: typeof TEMPLATES[0]) => {
    const exampleData = getExampleCV(template.id);
    if (exampleData) {
      onSelectTemplate(template, exampleData);
    }
  };

  return (
    <div className="space-y-6">
      {/* Filters */}
      <div className="space-y-4">
        <div>
          <h3 className="text-sm font-medium mb-2">Filter by Industry</h3>
          <div className="flex flex-wrap gap-2">
            <Button
              variant={selectedIndustry === null ? 'default' : 'outline'}
              size="sm"
              onClick={() => setSelectedIndustry(null)}
            >
              All
            </Button>
            {INDUSTRIES.map((industry) => (
              <Button
                key={industry.value}
                variant={selectedIndustry === industry.value ? 'default' : 'outline'}
                size="sm"
                onClick={() => setSelectedIndustry(industry.value)}
                className="gap-2"
              >
                {industryIcons[industry.value]}
                {industry.label}
              </Button>
            ))}
          </div>
        </div>

        <div>
          <h3 className="text-sm font-medium mb-2">Filter by Experience Level</h3>
          <div className="flex flex-wrap gap-2">
            <Button
              variant={selectedLevel === null ? 'default' : 'outline'}
              size="sm"
              onClick={() => setSelectedLevel(null)}
            >
              All Levels
            </Button>
            {EXPERIENCE_LEVELS.map((level) => (
              <Button
                key={level.value}
                variant={selectedLevel === level.value ? 'default' : 'outline'}
                size="sm"
                onClick={() => setSelectedLevel(level.value)}
              >
                {level.label}
              </Button>
            ))}
          </div>
        </div>
      </div>

      {/* Templates Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {filteredTemplates.map((template) => (
          <Card key={template.id} className="hover:shadow-lg transition-shadow cursor-pointer group">
            <CardHeader className="pb-3">
              <div className="flex items-center justify-between">
                <div className="p-2 bg-primary/10 rounded-lg group-hover:bg-primary/20 transition-colors">
                  {industryIcons[template.industry]}
                </div>
                <Badge variant="secondary" className="text-xs">
                  {EXPERIENCE_LEVELS.find(l => l.value === template.experienceLevel)?.label}
                </Badge>
              </div>
              <CardTitle className="text-lg mt-2">{template.name}</CardTitle>
              <CardDescription className="text-sm line-clamp-2">
                {template.description}
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-muted-foreground mb-1">Key Keywords Included:</p>
                  <div className="flex flex-wrap gap-1">
                    {template.keywords.slice(0, 4).map((keyword, idx) => (
                      <span
                        key={idx}
                        className="text-xs bg-secondary px-2 py-0.5 rounded-full"
                      >
                        {keyword}
                      </span>
                    ))}
                    {template.keywords.length > 4 && (
                      <span className="text-xs text-muted-foreground">
                        +{template.keywords.length - 4} more
                      </span>
                    )}
                  </div>
                </div>
                <Button
                  onClick={() => handleSelectTemplate(template)}
                  className="w-full gap-2"
                >
                  <Sparkles className="h-4 w-4" />
                  Use This Template
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {filteredTemplates.length === 0 && (
        <div className="text-center py-12">
          <p className="text-muted-foreground">
            No templates found for the selected filters. Try adjusting your criteria.
          </p>
        </div>
      )}
    </div>
  );
}
