import { Button } from '@/components/ui/button';
import { Download, FileText } from 'lucide-react';
import type { CVData } from '@/types/cv';
import jsPDF from 'jspdf';
import html2canvas from 'html2canvas';
import { Document, Paragraph, TextRun, HeadingLevel, Packer } from 'docx';
import { saveAs } from 'file-saver';

interface ExportButtonsProps {
  cvData: CVData;
  previewRef: React.RefObject<HTMLDivElement | null>;
}

export function ExportButtons({ cvData, previewRef }: ExportButtonsProps) {
  const exportToPDF = async () => {
    if (!previewRef.current) return;

    const element = previewRef.current;
    const canvas = await html2canvas(element, {
      scale: 2,
      useCORS: true,
      logging: false,
    });

    const imgData = canvas.toDataURL('image/png');
    const pdf = new jsPDF('p', 'mm', 'a4');
    const pdfWidth = pdf.internal.pageSize.getWidth();
    const pdfHeight = pdf.internal.pageSize.getHeight();
    const imgWidth = canvas.width;
    const imgHeight = canvas.height;
    const ratio = Math.min(pdfWidth / imgWidth, pdfHeight / imgHeight);

    pdf.addImage(
      imgData,
      'PNG',
      0,
      0,
      imgWidth * ratio,
      imgHeight * ratio
    );

    const fileName = `${cvData.personalInfo.firstName}_${cvData.personalInfo.lastName}_CV.pdf`;
    pdf.save(fileName);
  };

  const exportToDOCX = async () => {

    const doc = new Document({
      sections: [{
        properties: {},
        children: [
          // Header
          new Paragraph({
            text: `${cvData.personalInfo.firstName} ${cvData.personalInfo.lastName}`.toUpperCase(),
            heading: HeadingLevel.TITLE,
            spacing: { after: 200 },
          }),
          new Paragraph({
            children: [
              new TextRun({ text: cvData.personalInfo.email, bold: true }),
              new TextRun({ text: ' | ' }),
              new TextRun({ text: cvData.personalInfo.phone }),
            ],
            spacing: { after: 100 },
          }),
          new Paragraph({
            children: [
              new TextRun({ text: cvData.personalInfo.location }),
              ...(cvData.personalInfo.linkedin ? [
                new TextRun({ text: ' | ' }),
                new TextRun({ text: cvData.personalInfo.linkedin }),
              ] : []),
              ...(cvData.personalInfo.portfolio ? [
                new TextRun({ text: ' | ' }),
                new TextRun({ text: cvData.personalInfo.portfolio }),
              ] : []),
            ],
            spacing: { after: 300 },
          }),

          // Professional Summary
          ...(cvData.professionalSummary.summary || cvData.professionalSummary.rightToWork ? [
            new Paragraph({
              text: 'PROFESSIONAL SUMMARY',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            ...(cvData.professionalSummary.summary ? [
              new Paragraph({
                text: cvData.professionalSummary.summary,
                spacing: { after: 100 },
              }),
            ] : []),
            ...(cvData.professionalSummary.rightToWork ? [
              new Paragraph({
                children: [
                  new TextRun({ text: 'Right to Work: ', bold: true }),
                  new TextRun({ text: cvData.professionalSummary.rightToWork, italics: true }),
                ],
                spacing: { after: 200 },
              }),
            ] : []),
          ] : []),

          // Skills
          ...(cvData.skills.length > 0 && cvData.skills.some(s => s.category && s.items.length > 0) ? [
            new Paragraph({
              text: 'KEY SKILLS',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            ...cvData.skills.filter(s => s.category && s.items.length > 0).flatMap(skillCategory => [
              new Paragraph({
                children: [
                  new TextRun({ text: `${skillCategory.category}: `, bold: true }),
                  new TextRun({ text: skillCategory.items.join(', ') }),
                ],
                spacing: { after: 100 },
              }),
            ]),
          ] : []),

          // Work Experience
          ...(cvData.workExperience.length > 0 ? [
            new Paragraph({
              text: 'WORK EXPERIENCE',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            ...cvData.workExperience.flatMap(exp => [
              new Paragraph({
                children: [
                  new TextRun({ text: exp.jobTitle, bold: true }),
                ],
                spacing: { before: 100 },
              }),
              new Paragraph({
                children: [
                  new TextRun({ text: exp.company }),
                  new TextRun({ text: ` | ${exp.location}` }),
                ],
              }),
              new Paragraph({
                children: [
                  new TextRun({ 
                    text: `${formatDate(exp.startDate)} - ${exp.current ? 'Present' : formatDate(exp.endDate)}`,
                    italics: true,
                  }),
                ],
                spacing: { after: 100 },
              }),
              ...exp.responsibilities.filter(r => r.trim()).map(resp =>
                new Paragraph({
                  text: `• ${resp}`,
                  spacing: { after: 50 },
                })
              ),
            ]),
          ] : []),

          // Education
          ...(cvData.education.length > 0 ? [
            new Paragraph({
              text: 'EDUCATION',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            ...cvData.education.flatMap(edu => [
              new Paragraph({
                children: [
                  new TextRun({ text: edu.degree, bold: true }),
                ],
                spacing: { before: 100 },
              }),
              new Paragraph({
                text: edu.institution,
              }),
              new Paragraph({
                children: [
                  new TextRun({ text: edu.location }),
                  new TextRun({ text: ` | ${formatDate(edu.graduationDate)}` }),
                ],
                spacing: { after: 100 },
              }),
            ]),
          ] : []),

          // Certifications
          ...(cvData.certifications.length > 0 && cvData.certifications.some(c => c.name) ? [
            new Paragraph({
              text: 'CERTIFICATIONS',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            ...cvData.certifications.filter(c => c.name).map(cert =>
              new Paragraph({
                children: [
                  new TextRun({ text: cert.name, bold: true }),
                  ...(cert.issuingBody ? [new TextRun({ text: ` - ${cert.issuingBody}` })] : []),
                  ...(cert.dateObtained ? [new TextRun({ text: ` (${formatDate(cert.dateObtained)})`, italics: true })] : []),
                ],
                spacing: { after: 50 },
              })
            ),
          ] : []),

          // Languages
          ...(cvData.languages && cvData.languages.length > 0 ? [
            new Paragraph({
              text: 'LANGUAGES',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            new Paragraph({
              text: cvData.languages.join(' | '),
              spacing: { after: 200 },
            }),
          ] : []),

          // References
          ...(cvData.references ? [
            new Paragraph({
              text: 'REFERENCES',
              heading: HeadingLevel.HEADING_2,
              spacing: { before: 200, after: 100 },

            }),
            new Paragraph({
              text: cvData.references,
            }),
          ] : []),
        ],
      }],
    });

    const blob = await Packer.toBlob(doc);
    const fileName = `${cvData.personalInfo.firstName}_${cvData.personalInfo.lastName}_CV.docx`;
    saveAs(blob, fileName);
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return '';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-GB', { month: 'short', year: 'numeric' });
  };

  return (
    <div className="flex gap-2">
      <Button
        onClick={exportToPDF}
        variant="outline"
        className="gap-2"
        disabled={!cvData.personalInfo.firstName}
      >
        <Download className="h-4 w-4" />
        Export PDF
      </Button>
      <Button
        onClick={exportToDOCX}
        variant="outline"
        className="gap-2"
        disabled={!cvData.personalInfo.firstName}
      >
        <FileText className="h-4 w-4" />
        Export DOCX
      </Button>
    </div>
  );
}
