import type { CVData } from '@/types/cv';
import { forwardRef } from 'react';

interface CVPreviewProps {
  data: CVData;
  template?: string;
}

export const CVPreview = forwardRef<HTMLDivElement, CVPreviewProps>(
  ({ data }, ref) => {
    const formatDate = (dateString: string) => {
      if (!dateString) return '';
      const date = new Date(dateString);
      return date.toLocaleDateString('en-GB', { month: 'short', year: 'numeric' });
    };

    return (
      <div
        ref={ref}
        className="bg-white text-black p-8 min-h-[297mm] w-[210mm] mx-auto shadow-lg"
        style={{
          fontFamily: 'Arial, Helvetica, sans-serif',
          fontSize: '11pt',
          lineHeight: '1.4',
        }}
      >
        {/* Header */}
        <header className="border-b-2 border-gray-800 pb-4 mb-4">
          <h1 className="text-2xl font-bold text-gray-900 uppercase tracking-wide">
            {data.personalInfo.firstName} {data.personalInfo.lastName}
          </h1>
          <div className="text-sm text-gray-700 mt-2 space-y-1">
            <div className="flex flex-wrap gap-x-4">
              {data.personalInfo.email && (
                <span>{data.personalInfo.email}</span>
              )}
              {data.personalInfo.phone && (
                <span>{data.personalInfo.phone}</span>
              )}
            </div>
            <div className="flex flex-wrap gap-x-4">
              {data.personalInfo.location && (
                <span>{data.personalInfo.location}</span>
              )}
              {data.personalInfo.linkedin && (
                <span>{data.personalInfo.linkedin}</span>
              )}
              {data.personalInfo.portfolio && (
                <span>{data.personalInfo.portfolio}</span>
              )}
            </div>
          </div>
        </header>

        {/* Professional Summary */}
        {(data.professionalSummary.summary || data.professionalSummary.rightToWork) && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Professional Summary
            </h2>
            {data.professionalSummary.summary && (
              <p className="text-sm text-gray-800 mb-2">{data.professionalSummary.summary}</p>
            )}
            {data.professionalSummary.rightToWork && (
              <p className="text-sm text-gray-700 italic">
                <strong>Right to Work:</strong> {data.professionalSummary.rightToWork}
              </p>
            )}
          </section>
        )}

        {/* Skills */}
        {data.skills.length > 0 && data.skills.some(s => s.category && s.items.length > 0) && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Key Skills
            </h2>
            <div className="space-y-1">
              {data.skills.map((skillCategory, idx) => (
                skillCategory.category && skillCategory.items.length > 0 && (
                  <div key={idx} className="text-sm">
                    <span className="font-semibold">{skillCategory.category}:</span>{' '}
                    <span className="text-gray-800">{skillCategory.items.join(', ')}</span>
                  </div>
                )
              ))}
            </div>
          </section>
        )}

        {/* Work Experience */}
        {data.workExperience.length > 0 && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Work Experience
            </h2>
            <div className="space-y-3">
              {data.workExperience.map((exp) => (
                <div key={exp.id}>
                  <div className="flex justify-between items-start">
                    <div>
                      <h3 className="font-semibold text-sm">{exp.jobTitle}</h3>
                      <p className="text-sm text-gray-700">{exp.company}</p>
                    </div>
                    <div className="text-right text-sm text-gray-600">
                      <p>{exp.location}</p>
                      <p>
                        {formatDate(exp.startDate)} - {exp.current ? 'Present' : formatDate(exp.endDate)}
                      </p>
                    </div>
                  </div>
                  {exp.responsibilities.length > 0 && (
                    <ul className="list-disc list-inside text-sm text-gray-800 mt-1 ml-2">
                      {exp.responsibilities.filter(r => r.trim()).map((resp, idx) => (
                        <li key={idx}>{resp}</li>
                      ))}
                    </ul>
                  )}
                </div>
              ))}
            </div>
          </section>
        )}

        {/* Education */}
        {data.education.length > 0 && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Education
            </h2>
            <div className="space-y-2">
              {data.education.map((edu) => (
                <div key={edu.id}>
                  <div className="flex justify-between items-start">
                    <div>
                      <h3 className="font-semibold text-sm">{edu.degree}</h3>
                      <p className="text-sm text-gray-700">{edu.institution}</p>
                    </div>
                    <div className="text-right text-sm text-gray-600">
                      <p>{edu.location}</p>
                      <p>{formatDate(edu.graduationDate)}</p>
                    </div>
                  </div>
                  {edu.grade && (
                    <p className="text-sm text-gray-700">Grade: {edu.grade}</p>
                  )}
                  {edu.relevantModules && (
                    <p className="text-sm text-gray-600">Relevant Modules: {edu.relevantModules}</p>
                  )}
                </div>
              ))}
            </div>
          </section>
        )}

        {/* Certifications */}
        {data.certifications.length > 0 && data.certifications.some(c => c.name) && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Certifications
            </h2>
            <div className="space-y-1">
              {data.certifications.filter(c => c.name).map((cert) => (
                <div key={cert.id} className="text-sm">
                  <span className="font-semibold">{cert.name}</span>
                  {cert.issuingBody && <span> - {cert.issuingBody}</span>}
                  {cert.dateObtained && (
                    <span className="text-gray-600"> ({formatDate(cert.dateObtained)})</span>
                  )}
                </div>
              ))}
            </div>
          </section>
        )}

        {/* Languages */}
        {data.languages && data.languages.length > 0 && (
          <section className="mb-4">
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              Languages
            </h2>
            <p className="text-sm text-gray-800">{data.languages.join(' | ')}</p>
          </section>
        )}

        {/* References */}
        {data.references && (
          <section>
            <h2 className="text-sm font-bold text-gray-900 uppercase border-b border-gray-400 pb-1 mb-2">
              References
            </h2>
            <p className="text-sm text-gray-700">{data.references}</p>
          </section>
        )}
      </div>
    );
  }
);

CVPreview.displayName = 'CVPreview';
