import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Plus, Trash2, Briefcase, GraduationCap, Award, User, FileText } from 'lucide-react';
import type { CVData, WorkExperience, Education, Certification } from '@/types/cv';
import { VISA_TYPES } from '@/types/cv';

interface CVFormProps {
  data: CVData;
  onChange: (data: CVData) => void;
}

export function CVForm({ data, onChange }: CVFormProps) {
  const [activeTab, setActiveTab] = useState('personal');

  const updatePersonalInfo = (field: string, value: string) => {
    onChange({
      ...data,
      personalInfo: { ...data.personalInfo, [field]: value },
    });
  };

  const updateProfessionalSummary = (field: string, value: string) => {
    onChange({
      ...data,
      professionalSummary: { ...data.professionalSummary, [field]: value },
    });
  };

  const addWorkExperience = () => {
    const newExperience: WorkExperience = {
      id: Date.now().toString(),
      jobTitle: '',
      company: '',
      location: '',
      startDate: '',
      endDate: '',
      current: false,
      responsibilities: [''],
      achievements: [''],
    };
    onChange({
      ...data,
      workExperience: [...data.workExperience, newExperience],
    });
  };

  const updateWorkExperience = (id: string, field: string, value: any) => {
    onChange({
      ...data,
      workExperience: data.workExperience.map(exp =>
        exp.id === id ? { ...exp, [field]: value } : exp
      ),
    });
  };

  const removeWorkExperience = (id: string) => {
    onChange({
      ...data,
      workExperience: data.workExperience.filter(exp => exp.id !== id),
    });
  };

  const addResponsibility = (expId: string) => {
    const exp = data.workExperience.find(e => e.id === expId);
    if (exp) {
      updateWorkExperience(expId, 'responsibilities', [...exp.responsibilities, '']);
    }
  };

  const updateResponsibility = (expId: string, index: number, value: string) => {
    const exp = data.workExperience.find(e => e.id === expId);
    if (exp) {
      const newResponsibilities = [...exp.responsibilities];
      newResponsibilities[index] = value;
      updateWorkExperience(expId, 'responsibilities', newResponsibilities);
    }
  };

  const removeResponsibility = (expId: string, index: number) => {
    const exp = data.workExperience.find(e => e.id === expId);
    if (exp) {
      const newResponsibilities = exp.responsibilities.filter((_, i) => i !== index);
      updateWorkExperience(expId, 'responsibilities', newResponsibilities);
    }
  };

  const addEducation = () => {
    const newEducation: Education = {
      id: Date.now().toString(),
      degree: '',
      institution: '',
      location: '',
      graduationDate: '',
    };
    onChange({
      ...data,
      education: [...data.education, newEducation],
    });
  };

  const updateEducation = (id: string, field: string, value: string) => {
    onChange({
      ...data,
      education: data.education.map(edu =>
        edu.id === id ? { ...edu, [field]: value } : edu
      ),
    });
  };

  const removeEducation = (id: string) => {
    onChange({
      ...data,
      education: data.education.filter(edu => edu.id !== id),
    });
  };

  const addCertification = () => {
    const newCert: Certification = {
      id: Date.now().toString(),
      name: '',
      issuingBody: '',
      dateObtained: '',
    };
    onChange({
      ...data,
      certifications: [...data.certifications, newCert],
    });
  };

  const updateCertification = (id: string, field: string, value: string) => {
    onChange({
      ...data,
      certifications: data.certifications.map(cert =>
        cert.id === id ? { ...cert, [field]: value } : cert
      ),
    });
  };

  const removeCertification = (id: string) => {
    onChange({
      ...data,
      certifications: data.certifications.filter(cert => cert.id !== id),
    });
  };

  const addSkillCategory = () => {
    onChange({
      ...data,
      skills: [...data.skills, { category: '', items: [''] }],
    });
  };

  const updateSkillCategory = (index: number, field: string, value: any) => {
    const newSkills = [...data.skills];
    newSkills[index] = { ...newSkills[index], [field]: value };
    onChange({ ...data, skills: newSkills });
  };

  const addSkillItem = (categoryIndex: number) => {
    const newSkills = [...data.skills];
    newSkills[categoryIndex].items.push('');
    onChange({ ...data, skills: newSkills });
  };

  const updateSkillItem = (categoryIndex: number, itemIndex: number, value: string) => {
    const newSkills = [...data.skills];
    newSkills[categoryIndex].items[itemIndex] = value;
    onChange({ ...data, skills: newSkills });
  };

  const removeSkillItem = (categoryIndex: number, itemIndex: number) => {
    const newSkills = [...data.skills];
    newSkills[categoryIndex].items = newSkills[categoryIndex].items.filter((_, i) => i !== itemIndex);
    onChange({ ...data, skills: newSkills });
  };

  const removeSkillCategory = (index: number) => {
    onChange({
      ...data,
      skills: data.skills.filter((_, i) => i !== index),
    });
  };

  return (
    <div className="space-y-6">
      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid grid-cols-5 w-full">
          <TabsTrigger value="personal" className="flex items-center gap-2">
            <User className="h-4 w-4" />
            <span className="hidden sm:inline">Personal</span>
          </TabsTrigger>
          <TabsTrigger value="summary" className="flex items-center gap-2">
            <FileText className="h-4 w-4" />
            <span className="hidden sm:inline">Summary</span>
          </TabsTrigger>
          <TabsTrigger value="experience" className="flex items-center gap-2">
            <Briefcase className="h-4 w-4" />
            <span className="hidden sm:inline">Experience</span>
          </TabsTrigger>
          <TabsTrigger value="education" className="flex items-center gap-2">
            <GraduationCap className="h-4 w-4" />
            <span className="hidden sm:inline">Education</span>
          </TabsTrigger>
          <TabsTrigger value="skills" className="flex items-center gap-2">
            <Award className="h-4 w-4" />
            <span className="hidden sm:inline">Skills</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="personal" className="space-y-4 mt-4">
          <Card>
            <CardHeader>
              <CardTitle>Personal Information</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="firstName">First Name *</Label>
                  <Input
                    id="firstName"
                    value={data.personalInfo.firstName}
                    onChange={(e) => updatePersonalInfo('firstName', e.target.value)}
                    placeholder="John"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="lastName">Last Name *</Label>
                  <Input
                    id="lastName"
                    value={data.personalInfo.lastName}
                    onChange={(e) => updatePersonalInfo('lastName', e.target.value)}
                    placeholder="Smith"
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="email">Email *</Label>
                <Input
                  id="email"
                  type="email"
                  value={data.personalInfo.email}
                  onChange={(e) => updatePersonalInfo('email', e.target.value)}
                  placeholder="john.smith@email.com"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="phone">Phone *</Label>
                <Input
                  id="phone"
                  value={data.personalInfo.phone}
                  onChange={(e) => updatePersonalInfo('phone', e.target.value)}
                  placeholder="+44 7700 900000"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="location">Location *</Label>
                <Input
                  id="location"
                  value={data.personalInfo.location}
                  onChange={(e) => updatePersonalInfo('location', e.target.value)}
                  placeholder="London, UK"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="linkedin">LinkedIn (optional)</Label>
                <Input
                  id="linkedin"
                  value={data.personalInfo.linkedin}
                  onChange={(e) => updatePersonalInfo('linkedin', e.target.value)}
                  placeholder="linkedin.com/in/johnsmith"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="portfolio">Portfolio/Website (optional)</Label>
                <Input
                  id="portfolio"
                  value={data.personalInfo.portfolio}
                  onChange={(e) => updatePersonalInfo('portfolio', e.target.value)}
                  placeholder="johnsmith.com"
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="summary" className="space-y-4 mt-4">
          <Card>
            <CardHeader>
              <CardTitle>Professional Summary</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="summary">Professional Summary *</Label>
                <Textarea
                  id="summary"
                  value={data.professionalSummary.summary}
                  onChange={(e) => updateProfessionalSummary('summary', e.target.value)}
                  placeholder="Experienced professional with X years in..."
                  rows={5}
                />
                <p className="text-sm text-muted-foreground">
                  Write 3-4 sentences highlighting your experience, key skills, and what you bring to the role.
                </p>
              </div>
              <div className="space-y-2">
                <Label htmlFor="visaStatus">Visa Type</Label>
                <Select
                  value={data.professionalSummary.visaStatus}
                  onValueChange={(value) => updateProfessionalSummary('visaStatus', value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select visa type" />
                  </SelectTrigger>
                  <SelectContent>
                    {VISA_TYPES.map((visa) => (
                      <SelectItem key={visa.value} value={visa.label}>
                        {visa.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="rightToWork">Right to Work Statement *</Label>
                <Textarea
                  id="rightToWork"
                  value={data.professionalSummary.rightToWork}
                  onChange={(e) => updateProfessionalSummary('rightToWork', e.target.value)}
                  placeholder="Eligible for Skilled Worker visa sponsorship. Seeking opportunities with licensed UK sponsors."
                  rows={3}
                />
                <p className="text-sm text-muted-foreground">
                  Clearly state your visa status and sponsorship requirements.
                </p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="experience" className="space-y-4 mt-4">
          <div className="flex justify-between items-center">
            <h3 className="text-lg font-semibold">Work Experience</h3>
            <Button onClick={addWorkExperience} size="sm" className="gap-2">
              <Plus className="h-4 w-4" /> Add Experience
            </Button>
          </div>
          {data.workExperience.map((exp) => (
            <Card key={exp.id}>
              <CardContent className="pt-6 space-y-4">
                <div className="flex justify-end">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => removeWorkExperience(exp.id)}
                    className="text-destructive"
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>Job Title</Label>
                    <Input
                      value={exp.jobTitle}
                      onChange={(e) => updateWorkExperience(exp.id, 'jobTitle', e.target.value)}
                      placeholder="Software Engineer"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Company</Label>
                    <Input
                      value={exp.company}
                      onChange={(e) => updateWorkExperience(exp.id, 'company', e.target.value)}
                      placeholder="Company Name"
                    />
                  </div>
                </div>
                <div className="space-y-2">
                  <Label>Location</Label>
                  <Input
                    value={exp.location}
                    onChange={(e) => updateWorkExperience(exp.id, 'location', e.target.value)}
                    placeholder="London, UK"
                  />
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>Start Date</Label>
                    <Input
                      type="month"
                      value={exp.startDate}
                      onChange={(e) => updateWorkExperience(exp.id, 'startDate', e.target.value)}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>End Date</Label>
                    <Input
                      type="month"
                      value={exp.endDate}
                      onChange={(e) => updateWorkExperience(exp.id, 'endDate', e.target.value)}
                      disabled={exp.current}
                    />
                  </div>
                </div>
                <div className="flex items-center gap-2">
                  <Checkbox
                    checked={exp.current}
                    onCheckedChange={(checked) => updateWorkExperience(exp.id, 'current', checked)}
                  />
                  <Label className="text-sm">Current Position</Label>
                </div>
                <div className="space-y-2">
                  <Label>Responsibilities</Label>
                  {exp.responsibilities.map((resp, idx) => (
                    <div key={idx} className="flex gap-2">
                      <Input
                        value={resp}
                        onChange={(e) => updateResponsibility(exp.id, idx, e.target.value)}
                        placeholder="Describe a responsibility..."
                      />
                      <Button
                        variant="ghost"
                        size="icon"
                        onClick={() => removeResponsibility(exp.id, idx)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => addResponsibility(exp.id)}
                    className="gap-2"
                  >
                    <Plus className="h-4 w-4" /> Add Responsibility
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="education" className="space-y-4 mt-4">
          <div className="flex justify-between items-center">
            <h3 className="text-lg font-semibold">Education</h3>
            <Button onClick={addEducation} size="sm" className="gap-2">
              <Plus className="h-4 w-4" /> Add Education
            </Button>
          </div>
          {data.education.map((edu) => (
            <Card key={edu.id}>
              <CardContent className="pt-6 space-y-4">
                <div className="flex justify-end">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => removeEducation(edu.id)}
                    className="text-destructive"
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
                <div className="space-y-2">
                  <Label>Degree/Qualification</Label>
                  <Input
                    value={edu.degree}
                    onChange={(e) => updateEducation(edu.id, 'degree', e.target.value)}
                    placeholder="Bachelor of Science in Computer Science"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Institution</Label>
                  <Input
                    value={edu.institution}
                    onChange={(e) => updateEducation(edu.id, 'institution', e.target.value)}
                    placeholder="University Name"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Location</Label>
                  <Input
                    value={edu.location}
                    onChange={(e) => updateEducation(edu.id, 'location', e.target.value)}
                    placeholder="London, UK"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Graduation Date</Label>
                  <Input
                    type="month"
                    value={edu.graduationDate}
                    onChange={(e) => updateEducation(edu.id, 'graduationDate', e.target.value)}
                  />
                </div>
                <div className="space-y-2">
                  <Label>Grade/Classification (optional)</Label>
                  <Input
                    value={edu.grade || ''}
                    onChange={(e) => updateEducation(edu.id, 'grade', e.target.value)}
                    placeholder="First Class Honours"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Relevant Modules (optional)</Label>
                  <Input
                    value={edu.relevantModules || ''}
                    onChange={(e) => updateEducation(edu.id, 'relevantModules', e.target.value)}
                    placeholder="Software Engineering, Data Structures..."
                  />
                </div>
              </CardContent>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="skills" className="space-y-4 mt-4">
          <Card>
            <CardHeader>
              <CardTitle>Certifications</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <Button onClick={addCertification} size="sm" className="gap-2">
                <Plus className="h-4 w-4" /> Add Certification
              </Button>
              {data.certifications.map((cert) => (
                <div key={cert.id} className="grid grid-cols-12 gap-2 items-end">
                  <div className="col-span-4">
                    <Input
                      value={cert.name}
                      onChange={(e) => updateCertification(cert.id, 'name', e.target.value)}
                      placeholder="Certification Name"
                    />
                  </div>
                  <div className="col-span-3">
                    <Input
                      value={cert.issuingBody}
                      onChange={(e) => updateCertification(cert.id, 'issuingBody', e.target.value)}
                      placeholder="Issuing Body"
                    />
                  </div>
                  <div className="col-span-3">
                    <Input
                      type="month"
                      value={cert.dateObtained}
                      onChange={(e) => updateCertification(cert.id, 'dateObtained', e.target.value)}
                      placeholder="Date"
                    />
                  </div>
                  <div className="col-span-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => removeCertification(cert.id)}
                      className="text-destructive"
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Skills</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <Button onClick={addSkillCategory} size="sm" className="gap-2">
                <Plus className="h-4 w-4" /> Add Skill Category
              </Button>
              {data.skills.map((skillCategory, catIdx) => (
                <div key={catIdx} className="border rounded-lg p-4 space-y-3">
                  <div className="flex justify-between items-center">
                    <Input
                      value={skillCategory.category}
                      onChange={(e) => updateSkillCategory(catIdx, 'category', e.target.value)}
                      placeholder="Category (e.g., Programming Languages)"
                      className="w-1/2"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => removeSkillCategory(catIdx)}
                      className="text-destructive"
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                  <div className="space-y-2">
                    {skillCategory.items.map((item, itemIdx) => (
                      <div key={itemIdx} className="flex gap-2">
                        <Input
                          value={item}
                          onChange={(e) => updateSkillItem(catIdx, itemIdx, e.target.value)}
                          placeholder="Skill name"
                        />
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => removeSkillItem(catIdx, itemIdx)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    ))}
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => addSkillItem(catIdx)}
                      className="gap-2"
                    >
                      <Plus className="h-4 w-4" /> Add Skill
                    </Button>
                  </div>
                </div>
              ))}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Languages</CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                value={data.languages?.join('\n') || ''}
                onChange={(e) => onChange({ ...data, languages: e.target.value.split('\n').filter(Boolean) })}
                placeholder="English (Professional Working Proficiency)&#10;Spanish (Native)&#10;French (Conversational)"
                rows={4}
              />
              <p className="text-sm text-muted-foreground mt-2">
                Enter each language on a new line with proficiency level.
              </p>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
