import { useState, useRef } from 'react';
import { Navigation } from '@/sections/Navigation';
import { Hero } from '@/sections/Hero';
import { Features } from '@/sections/Features';
import { Templates } from '@/sections/Templates';
import { HowItWorks } from '@/sections/HowItWorks';
import { Testimonials } from '@/sections/Testimonials';
import { FAQ } from '@/sections/FAQ';
import { CTA } from '@/sections/CTA';
import { Footer } from '@/sections/Footer';
import { CVForm } from '@/components/cv-builder/CVForm';
import { CVPreview } from '@/components/cv-builder/CVPreview';
import { ExportButtons } from '@/components/cv-builder/ExportButtons';
import { TemplateSelector } from '@/components/cv-builder/TemplateSelector';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { ArrowLeft, Eye, Edit3 } from 'lucide-react';
import type { CVData } from '@/types/cv';
import { INITIAL_CV_DATA } from '@/types/cv';
import { getExampleCV } from '@/data/templates';
import { Toaster } from '@/components/ui/sonner';
import { toast } from 'sonner';

type ViewState = 'home' | 'template-selection' | 'builder';

function App() {
  const [viewState, setViewState] = useState<ViewState>('home');
  const [cvData, setCvData] = useState<CVData>(INITIAL_CV_DATA);
  const [activeBuilderTab, setActiveBuilderTab] = useState('edit');
  const previewRef = useRef<HTMLDivElement>(null);

  const handleStartBuilding = () => {
    setViewState('template-selection');
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const handleSelectTemplate = (templateId: string) => {
    const exampleData = getExampleCV(templateId);
    if (exampleData) {
      setCvData(exampleData);
      toast.success(`Loaded ${templateId.replace(/-/g, ' ')} template!`, {
        description: 'You can now edit the template with your own information.',
      });
    }
    setViewState('builder');
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const handleTemplateSelectFromSelector = (template: any, exampleData: CVData) => {
    setCvData(exampleData);
    setViewState('builder');
    window.scrollTo({ top: 0, behavior: 'smooth' });
    toast.success(`Loaded ${template.name} template!`, {
      description: 'You can now edit the template with your own information.',
    });
  };

  const handleBackToHome = () => {
    setViewState('home');
    setCvData(INITIAL_CV_DATA);
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const handleBackToTemplates = () => {
    setViewState('template-selection');
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  // Home Page View
  if (viewState === 'home') {
    return (
      <div className="min-h-screen bg-background">
        <Navigation onStartBuilding={handleStartBuilding} />
        <main>
          <Hero onStartBuilding={handleStartBuilding} />
          <Features />
          <div id="templates">
            <Templates onSelectTemplate={handleSelectTemplate} />
          </div>
          <div id="how-it-works">
            <HowItWorks />
          </div>
          <div id="testimonials">
            <Testimonials />
          </div>
          <div id="faq">
            <FAQ />
          </div>
          <CTA onStartBuilding={handleStartBuilding} />
        </main>
        <Footer />
        <Toaster />
      </div>
    );
  }

  // Template Selection View
  if (viewState === 'template-selection') {
    return (
      <div className="min-h-screen bg-background">
        <Navigation onStartBuilding={handleStartBuilding} />
        <main className="py-8">
          <div className="container mx-auto px-4">
            <Button
              variant="ghost"
              onClick={handleBackToHome}
              className="mb-6 gap-2"
            >
              <ArrowLeft className="h-4 w-4" />
              Back to Home
            </Button>
            
            <div className="max-w-4xl mx-auto mb-8">
              <h1 className="text-3xl md:text-4xl font-bold mb-4">
                Choose a CV Template
              </h1>
              <p className="text-lg text-muted-foreground">
                Select an industry-specific template designed for UK visa sponsorship applications. 
                Each template includes ATS-optimized formatting and relevant keywords.
              </p>
            </div>

            <TemplateSelector onSelectTemplate={handleTemplateSelectFromSelector} />
          </div>
        </main>
        <Footer />
        <Toaster />
      </div>
    );
  }

  // Builder View
  return (
    <div className="min-h-screen bg-background">
      <Navigation onStartBuilding={handleStartBuilding} />
      <main className="py-8">
        <div className="container mx-auto px-4">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-6">
            <div className="flex items-center gap-4">
              <Button
                variant="ghost"
                onClick={handleBackToTemplates}
                className="gap-2"
              >
                <ArrowLeft className="h-4 w-4" />
                Back
              </Button>
              <div>
                <h1 className="text-2xl font-bold">CV Builder</h1>
                <p className="text-sm text-muted-foreground">
                  {cvData.personalInfo.firstName 
                    ? `Editing: ${cvData.personalInfo.firstName} ${cvData.personalInfo.lastName}`
                    : 'Create your ATS-friendly CV'}
                </p>
              </div>
            </div>
            <ExportButtons cvData={cvData} previewRef={previewRef} />
          </div>

          <Tabs value={activeBuilderTab} onValueChange={setActiveBuilderTab} className="w-full">
            <TabsList className="grid w-full grid-cols-2 mb-6">
              <TabsTrigger value="edit" className="gap-2">
                <Edit3 className="h-4 w-4" />
                Edit CV
              </TabsTrigger>
              <TabsTrigger value="preview" className="gap-2">
                <Eye className="h-4 w-4" />
                Preview
              </TabsTrigger>
            </TabsList>

            <TabsContent value="edit" className="mt-0">
              <div className="grid lg:grid-cols-2 gap-8">
                <div>
                  <CVForm data={cvData} onChange={setCvData} />
                </div>
                <div className="hidden lg:block">
                  <div className="sticky top-24">
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="font-semibold">Live Preview</h3>
                      <span className="text-xs text-muted-foreground">
                        Updates as you type
                      </span>
                    </div>
                    <div className="transform scale-[0.6] origin-top">
                      <CVPreview data={cvData} ref={previewRef} />
                    </div>
                  </div>
                </div>
              </div>
            </TabsContent>

            <TabsContent value="preview" className="mt-0">
              <div className="flex justify-center">
                <CVPreview data={cvData} ref={previewRef} />
              </div>
            </TabsContent>
          </Tabs>
        </div>
      </main>
      <Footer />
      <Toaster />
    </div>
  );
}

export default App;
